/**
* \file: trace_ald_interface.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* Implements functions for supporting Authorization Level in TRACE. Uses interface
* provided by the Authorization Level Daemon Plugin.
*
* \component: TRACE
*
* \author: Bewoayia Kebianyor
*
* \copyright (c) 2012, 2013 Advanced Driver Information Technology.

* This code is developed by Advanced Driver Information Technology.

* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.

* All rights reserved.
*
* \history
* 0.1 Bewoayia Kebianyor Initial version
*
***********************************************************************/

#ifdef TRACE_ALD_INTERFACE_ENABLED

#include <stdio.h>
#include <sys/poll.h>
#include <stdbool.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include "trace_base.h"
#include <search.h>
#include <adit-components/ald_plugin.h>

#define TRACE_ALD_CNFG_FILE "/etc/adit_trace_ald.conf"
#define TRACE_ALD_STRCPY_BUF_SIZE  1024


static void TRACE_ALD_interface_connection_event_handler(void *ptr,void *ptr2);
static void TRACE_ALD_interface_disconnect_event_handler(void * data_ptr, void * data_ptr2);
static void TRACE_ALD_interface_level_changed_event_handler(unsigned int level, void *ptr,void *ptr2);
static int TRACE_ALD_interface_read_config_file(TRACE_ALD_Mgr * ald_mgr);
static int TRACE_ALD_interface_init_trace_command_structure(TRACE_ALD_Mgr *ald_mgr, int security_level_pos, int command_type,int config_pos);
int TRACE_ALD_interface_open_conn_to_host(TRACE_mgr * trace_mgr);
int TRACE_ALD_interface_close_conn_to_host(TRACE_mgr * trace_mgr);
static bool TRACE_ALD_interface_is_traceBeCmd_Supported(void *mgr);
static bool TRACE_ALD_interface_is_classCfgSupported(void *mgr,TRACE_ALD_CmdTypeConfig *traceCfgData);
static bool TRACE_ALD_interface_cfg_ena_all(void *mgr,TRACE_ALD_CmdTypeConfig *traceCfgData);


static const ald_plugin_callbacks_t ald_plugin_callbacks=
{
    .connection_clbk          = TRACE_ALD_interface_connection_event_handler,
    .disConnection_clbk       = TRACE_ALD_interface_disconnect_event_handler,
    .calculate_log_level_clbk = TRACE_ALD_interface_level_changed_event_handler,
    .cmd_received_clbk        = NULL
};

void TRACE_ALD_interface_connection_event_handler(void *ptr,void *ptr2)
{
    if((ptr != NULL) && (ptr2 != NULL))
    {
        TRACE_ALD_interface_level_changed_event_handler(ald_plugin_get_security_level(1),ptr,ptr2);
    }
}


void TRACE_ALD_interface_disconnect_event_handler(void * data_ptr, void * data_ptr2)
{
    TRACE_ALD_Mgr *ald_mgr;

    if((data_ptr != NULL) && (data_ptr2 != NULL))
    {
        ald_mgr = (TRACE_ALD_Mgr *) data_ptr2;
        // Switch to default security level when no ALD is connected.
        TRACE_ALD_interface_level_changed_event_handler(ald_mgr->default_security_level,data_ptr,data_ptr2);
    }
}

void TRACE_ALD_interface_level_changed_event_handler(unsigned int level, void *ptr,void *ptr2)
{
     TRACE_ALD_Mgr *ald_mgr;
     TRACE_mgr * trace_mgr;

     ER ret = 0;

    if((ptr != NULL) && (ptr2 != NULL))
    {
        trace_mgr = (TRACE_mgr *)ptr;
        ald_mgr = (TRACE_ALD_Mgr *)ptr2;

        /* If the same security level received as previous, silently ignore this*/
        if(ald_mgr->system_security_level == level) return;

        if((level >= ald_mgr->ald_customer_level) && (level < ald_mgr->ald_resident_level))
        {
            if(TRACE_ALD_interface_close_conn_to_host(trace_mgr) != 0)
                TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"TRACE_ALD_interface_level_changed_event_handler: Failed to close client connections\n");
        }
        else if((level >= ald_mgr->ald_resident_level) && (level < ald_mgr->ald_service_level))
        {
            if(ald_mgr->system_security_level < ald_mgr->ald_resident_level)
            {
                /* Only open client connections if previously closed*/
                ret = TRACE_ALD_interface_open_conn_to_host(trace_mgr);
                if( ret != E_OK)
                    TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"TRACE_ALD_interface_level_changed_event_handler: Failed to Open all client connections\n");
            }
        }
        else if((level >= ald_mgr->ald_service_level)&& (level < ald_mgr->ald_production_level))
        {
            if(ald_mgr->system_security_level < ald_mgr->ald_resident_level)
            {
                /* Only open client connections if previously closed*/
                ret = TRACE_ALD_interface_open_conn_to_host(trace_mgr);
                if( ret != E_OK)
                    TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"TRACE_ALD_interface_level_changed_event_handler: Failed to Open all client connections\n");
            }
        }
        else if(level >= ald_mgr->ald_production_level)
        {
            if(ald_mgr->system_security_level < ald_mgr->ald_resident_level)
            {
                //Only open client connections if previously closed
                ret = TRACE_ALD_interface_open_conn_to_host(trace_mgr);
                if( ret != E_OK)
                    TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"TRACE_ALD_interface_level_changed_event_handler: Failed to Open all client connections\n");
            }
        }
        else
        {
            TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"Invalid Security Level. Check settings in /etc/adit_trace_ald.conf file\n");
        }

        ald_mgr->system_security_level = level;
    }
}


int TRACE_ALD_interface_init_ald_plugin(TRACE_ALD_Mgr *ald_mgr)
{
    int ret  = 0;

    if(ald_mgr != NULL)
    {
        if(TRACE_ALD_interface_read_config_file(ald_mgr) < 0)
            ret = -1;

        if(ret == 0)
        {
            ret = ald_plugin_init(&ald_plugin_callbacks,ald_mgr->system_security_level);

            if(ret == 0)
            {
                /* Initialise daefault security level for TRACE*/
                ald_mgr->system_security_level = ald_plugin_get_security_level(1);

                ret = TRACE_ALD_interface_init_epoll_handle(ald_mgr);
            }
        }
        else
            TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"Failed to initialised ALD plugin in Func (TRACE_ALD_interface_int_ald_plugin)\n");

    }
    else
    {
        ret  = -1;
        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"Failed to initialised ALD plugin: NULL POINTER\n");
    }

    return ret;
}



int TRACE_ALD_interface_deinit_ald_plugin(void * trace_mgr,TRACE_ALD_Mgr *ald_mgr)
{
    int ret         = 0;
    int cnt         = 0;
    int cmd_cfg_cnt = 0;

    if((ald_mgr != NULL) && (trace_mgr != NULL))
    {
        close(ald_mgr->file_desc);


        TRACE_ALD_interface_deinit_epoll_handle(ald_mgr);

        for(cnt= 0; (cnt < TRACE_ALD_NUM_SECURITY_LEVELS);cnt ++)
        {
            for(cmd_cfg_cnt = 0; (cmd_cfg_cnt <TRACE_ALD_MAX_TRACE_COMMMAND_TYPES); cmd_cfg_cnt++)
            {
                // free allocated memory for the class specific confiuration parameters
                if((ald_mgr->aldConfigTable[cnt].traceCmdConfig[cmd_cfg_cnt].classCmdConfig) != NULL)
                {
                    free((ald_mgr->aldConfigTable[cnt].traceCmdConfig[cmd_cfg_cnt].classCmdConfig));
                }
                // free allocated memory for the sub command list
                if((ald_mgr->aldConfigTable[cnt].traceCmdConfig[cmd_cfg_cnt].subCmdList) != NULL)
                {
                   free (ald_mgr->aldConfigTable[cnt].traceCmdConfig[cmd_cfg_cnt].subCmdList);
                }
             }
         }

         ret = ald_plugin_deinit(trace_mgr,ald_mgr);


        /* Clean  up the data structure for ALD interface*/
        ald_mgr->file_desc = -1;
        // ald_mgr->config_state = TRACE_ALD_INTERFACE_NOT_INITIALISED;
        ald_mgr->system_security_level = 0;
        ald_mgr->default_security_level = 0;
        ald_mgr->ald_customer_level = 0;
        ald_mgr->ald_resident_level = 0;
        ald_mgr->ald_service_level = 0;
        ald_mgr->ald_production_level = 0;
        ald_mgr->numTotalBEConfig = 0;
    }
    else
    {
        fprintf(stdout," Failed to de-initialised ALD plugin: NULL POINTER\n");
    }

    return ret;
}


/**
* Initialise epoll Handle to  and add filedescriptors
*
* \param  mgr  Pointer to Trace manager
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
EXPORT ER TRACE_ALD_interface_init_epoll_handle(TRACE_ALD_Mgr *ald_mgr)
{

    if(ald_mgr == NULL)
        return -1;

     ald_mgr->epollHandle = -1;
    /* --- Creation of epoll instance handle begin */
    ald_mgr->epollHandle = epoll_create(TRACE_ALD_MAX_EPOLL_EVENTS);
    if(-1 == ald_mgr->epollHandle)
    {
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR,"Creation of epoll instance failed!\n");
        return -1;
    }
    else /* Iniitialise event_fd to receive events when level change in ALD plugin*/
    {
        ald_mgr->file_desc  = ald_plugin_get_pollfd();

        if(ald_mgr->file_desc > 0)
        {
            ald_mgr->eventHandle.events = EPOLLIN;
            ald_mgr->eventHandle.data.fd = ald_mgr->file_desc;

            if (epoll_ctl(ald_mgr->epollHandle, EPOLL_CTL_ADD, ald_mgr->file_desc, &ald_mgr->eventHandle) == -1)
            {
                return -1;
            }
        }
    }

    return E_OK;
}


/**
* Un Initialise epoll Handle to  and add filedescriptors
*
* \param  mgr  Pointer to Trace manager
*
* \return E_OK if successful
*              else error returned by concrete implementation
*/
EXPORT ER TRACE_ALD_interface_deinit_epoll_handle(TRACE_ALD_Mgr *ald_mgr)
{

    if(ald_mgr == NULL)
        return -1;

    close(ald_mgr->epollHandle);

    return E_OK;
}

EXPORT ER TRACE_ALD_interface_handle_ald_event(void *mgr,TRACE_ALD_Mgr *ald_mgr,bool ald_connected)
{
    TRACE_mgr * trace_mgr = (TRACE_mgr*)mgr;
    ER rc = E_OK;

    if((ald_mgr == NULL) || (trace_mgr == NULL))
        return -1;

    if(ald_connected == TRUE)
    {
        rc = ald_plugin_dispatch(trace_mgr, ald_mgr);
    }
    else
    {
        rc = TRACE_ALD_interface_close_conn_to_host(trace_mgr);
    }

    return rc;
}



int TRACE_ALD_interface_open_conn_to_host(TRACE_mgr *mgr)
{
    ER  rc = E_OK;

    if(mgr == NULL)
       return E_FAIL;
#ifndef TRACE_DLT_EXIST
        mgr->ctrl.cmd_info.rd_wait = FALSE;

        mgr->ctrl.swi_req = FALSE;

        if (NULL != g_TRACE_fnctbl.io_opn[mgr->sh->trc.d_cnfg.actv_chan])
        {
            rc = (g_TRACE_fnctbl.io_opn[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);
        }

        /* Wakeup dependent tasks */
        pthread_mutex_lock(&mgr->ctrl.condWait.condMutex);
        pthread_cond_broadcast(&mgr->ctrl.condWait.condVariable);
        pthread_mutex_unlock(&mgr->ctrl.condWait.condMutex);
        mgr->ctrl.swi_req = FALSE;
#endif
    return rc;
}


int TRACE_ALD_interface_close_conn_to_host(TRACE_mgr *mgr)
{

    ER  rc = E_OK;

    if(mgr == NULL)
       return -1;
#ifndef TRACE_DLT_EXIST
        /* Request to RD task to switch IO channel */
        mgr->ctrl.cmd_info.rd_wait = TRUE;

         /* Wait until RD task goes to wait */
        pthread_mutex_lock(&mgr->ctrl.condSleep.condMutex);
        rc = pthread_cond_wait(&mgr->ctrl.condSleep.condVariable,
                      &mgr->ctrl.condSleep.condMutex);
        pthread_mutex_unlock(&mgr->ctrl.condSleep.condMutex);

        pthread_mutex_lock(&mgr->ctrl.condSleep.condMutex);

        /* Request WR task to switch IO channel */

        TRACE_set_flag(&mgr->sh->evt_id[EVT_TRC_Q], TRACE_CHNL_SWTCH_WRT);

        /* Fix: when Trace Q is full, switch req is not serviced
         * set this flag too.. as write task might be looping in check_q()
         * using event flag has performance issue as it uses mutex/cond_wait
         */
        mgr->ctrl.swi_req = TRUE;

        /* wait until WR task goes to wait */
        rc = pthread_cond_wait(&mgr->ctrl.condSleep.condVariable,
                      &mgr->ctrl.condSleep.condMutex);
        pthread_mutex_unlock(&mgr->ctrl.condSleep.condMutex);

        /* Now close physical I/O device */
        rc = (g_TRACE_fnctbl.io_cls[mgr->sh->trc.d_cnfg.actv_chan])(mgr, NULL);
#endif

   return rc;
}


static int TRACE_ALD_interface_read_config_file(TRACE_ALD_Mgr *ald_mgr)
{

    int config_count                 = 0;
    int trace_cmd_pos                = -1;
    int classCfgPos                  = 0;
    int loop_cnt                     = 0;
    char* pch                        = NULL;
    const char* filename             = TRACE_ALD_CNFG_FILE;
    int continue_loop                = 1;
    int serviceLevelBECfgCmd_present = 0;
    int serviceLevelClassCfg_present = 0;
    int subCmdType_cnt               = 0;
    int default_security_level_set   = 0;
    int classID                      = 0;
    int traceLevel                   = 0;
    int traceCmdType                 = 0;
    int security_level               = -1;
    int ret_value                    = 0;
    char line[TRACE_ALD_STRCPY_BUF_SIZE]            = {0};
    char token[TRACE_ALD_STRCPY_BUF_SIZE]           = {0};
    char value[TRACE_ALD_STRCPY_BUF_SIZE]           = {0};
    char tmpbuf[TRACE_ALD_STRCPY_BUF_SIZE]          = {0};
    char SubCommandList[TRACE_ALD_STRCPY_BUF_SIZE]  = {0};

    ENTRY entry;
    ENTRY *retEntry;


    if((ald_mgr== NULL))
    {
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Invalid function parameters : TRACE_ALD_interface_read_config_file()\n");
        return -1;
    }

   char *list   = &ald_mgr->list[0][0];
   char *keybuf = &ald_mgr->keybuf[0][0];

    FILE* pFile = fopen(filename, "r");

    fprintf(stdout,"Reading ALD conf file\n");

    if(NULL != pFile)
    {
        while(continue_loop)
        {
            /* fetch line from configuration file */
            if(fgets(line, 1024, pFile) != NULL)
            {
                pch = strtok(line, " =\r\n");
                token[0] = 0;
                value[0] = 0;


                while(pch != NULL)
                {
                    if(strcmp(pch, "#") == 0)
                    {
                        break;
                    }
                    if(strcmp(pch, "[ServiceLevel_BackendConfigCmd]") == 0)
                    {
                        traceCmdType  =0;
                        strcpy(SubCommandList,"NULL");
                        serviceLevelBECfgCmd_present = 1;
                        break;
                    }
                    if(strcmp(pch, "[ServiceLevel_ClassConfigCmd]") == 0)
                    {
                        classID = 0;
                        traceLevel  = -1;
                        serviceLevelClassCfg_present = 1;
                        break;
                    }
                    if(token[0] == 0)
                    {
                        strncpy(token, pch, sizeof(token)-1);
                        token[sizeof(token)-1] = 0;
                    }
                    else
                    {
                        strncpy(value, pch, sizeof(value)-1);
                        value[sizeof(value)-1] = 0;
                        break;
                    }
                    pch = strtok(NULL, " =\r\n");
                }
                if(token[0] && value[0])
                {
                    if(strcmp(token, "DefaultSecurityLevel") == 0)
                    {
                        ald_mgr->default_security_level = atoi(value);
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"Configured Default Security Level: %d\n",ald_mgr->default_security_level);
                        default_security_level_set = 1;
                    }
                    else if(strcmp(token, "CustomerSecurityLevel") == 0)
                    {
                        ald_mgr->ald_customer_level = atoi(value);
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Configured Customer Security Level: %d\n",ald_mgr->ald_customer_level);
                    }
                    else if(strcmp(token, "ServiceSecurityLevel") == 0)
                    {
                        ald_mgr->ald_service_level = atoi(value);
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE,"Configured Service Security Level: %d\n",ald_mgr->ald_service_level);
                    }
                    else if(strcmp(token, "ResidentSecurityLevel") == 0)
                    {
                        ald_mgr->ald_resident_level = atoi(value);
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Configured Resident Security Level: %d\n",ald_mgr->ald_resident_level);
                    }
                    else if(strcmp(token, "ProductionSecurityLevel") == 0)
                    {
                        ald_mgr->ald_production_level = atoi(value);
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Configured Production Security Level: %d\n", ald_mgr->ald_production_level);
                    }
                    else if(strcmp(token,"CommandType")==0)
                    {
                        traceCmdType = atoi(value);
                    }
                    else if(strcmp(token,"SubCmdType")==0)
                    {
                        strncpy(SubCommandList,value,sizeof(SubCommandList)-1);
                        SubCommandList[sizeof(SubCommandList)-1] = 0;
                    }
                    else if(strcmp(token,"ClassID")==0)
                    {
                        classID = atoi(value);
                        /* in case of HEX values */
                        if((value[0] == '0') &&(value[1] == 'x'))
                        {
                           classID = strtol(value,NULL,16);
                        }
                    }
                    else if(strcmp(token,"TraceLevel")==0)
                    {
                        traceLevel = atoi(value);
                    }
                    else
                    {
                        TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Reading TRACE Security Level Config file: Unknown option: %s = %s\n", token, value);
                    }
                }//end (token[0] && value[0])
                //Verify if Application specific configuration completely scanned and store in table
                if((serviceLevelBECfgCmd_present == 1)&&(traceCmdType !=0) && (strcmp(SubCommandList,"NULL")!=0))
                {
                    security_level = ald_mgr->ald_service_level;
                    ret_value = TRACE_ALD_interface_init_trace_command_structure(ald_mgr,TRACE_ALD_SERVICE_LEVEL_POS,traceCmdType,config_count);
                    if( ret_value  != 0)
                    {
                        TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Could not initialise structures for reading TRACE security configuration\n");
                        TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "TRACE STARTUP failed.\n");
                        ald_mgr->system_security_level = 0;
                        fclose(pFile);
                        return ret_value;
                    }

                    subCmdType_cnt    = 0;

                    strncpy(tmpbuf,SubCommandList,sizeof(tmpbuf)-1);
                    tmpbuf[sizeof(tmpbuf)-1]= 0;

                    pch = strtok(tmpbuf,",");

                    if(pch != NULL)
                    {
                        if(strcmp(SubCommandList,pch)==0)
                        {
                            snprintf(&keybuf[config_count * TRACE_ALD_MAXFILTERSIZE],TRACE_ALD_MAXFILTERSIZE,"%d:%d;",security_level,traceCmdType);
                            ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].subCmdList[subCmdType_cnt] = atoi(SubCommandList);
                            ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].cmdType = traceCmdType;
                            ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].num_sub_cmds = subCmdType_cnt+1;
                        }
                        else
                        {
                            while(pch != NULL)
                            {
                                snprintf(&keybuf[config_count * TRACE_ALD_MAXFILTERSIZE],TRACE_ALD_MAXFILTERSIZE,"%d:%d;",security_level,traceCmdType);
                                ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].subCmdList[subCmdType_cnt] = atoi(pch);
                                ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].cmdType = traceCmdType;
                                pch = strtok (NULL,",");
                                subCmdType_cnt = subCmdType_cnt + 1;
                                ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[config_count].num_sub_cmds = subCmdType_cnt;
                            }
                        }

                        if(traceCmdType == TRACE_COMMAND)
                        {
                            // store the position where trace command is saved to be able to update the structure with class specific configuration.
                            trace_cmd_pos = config_count;
                        }
                        strcpy(SubCommandList,"NULL");
                        serviceLevelBECfgCmd_present = 0;
                        traceCmdType = 0;
                        config_count = config_count + 1;
                    }
                }
                if((serviceLevelClassCfg_present == 1)&&(traceLevel != -1) && (classID != 0)&&(trace_cmd_pos != -1))
                {
                    security_level = ald_mgr->ald_service_level;
                    ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[trace_cmd_pos].classCmdConfig[classCfgPos].trace_level = traceLevel;
                    ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[trace_cmd_pos].classCmdConfig[classCfgPos].classID = classID;
                    ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[trace_cmd_pos].num_class_config += 1;
                    classCfgPos = classCfgPos + 1;
                    serviceLevelClassCfg_present  = 0;
                    classID = 0;
                    traceLevel = -1;
                }

            } /* End of if(fgets(line, 1024, pFile) != NULL) */
            else
            {
                continue_loop = 0; /* To break from the while loop */
            }
        } /* End of while(continue_loop) */

        fclose(pFile);

        if(default_security_level_set == 0)
        {
             TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Invalid default Security (Authorization) Level Set for TRACE\n");
             TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Please check /etc/adit_trace_ald.cnf file again\n");
             TRACE_SYSLOG(TRACE_SYSLOG_NOTICE, "Default level internally set to 0 \n");
             ald_mgr->system_security_level = 0;
        }
        else
        {
            ald_mgr->system_security_level = ald_mgr->default_security_level;
            if(config_count >0)
            {
               if( hcreate(config_count) == 0)
               {
                  return -1;
               }

               for(loop_cnt=0;loop_cnt<config_count;loop_cnt++)
               {
                    entry.key = (char*)&keybuf[loop_cnt* TRACE_ALD_MAXFILTERSIZE];
                    entry.data = (void *)&ald_mgr->aldConfigTable[TRACE_ALD_SERVICE_LEVEL_POS].traceCmdConfig[loop_cnt];
                    retEntry = hsearch(entry, ENTER);
                    if(retEntry == NULL)
                    {
                        // free memory allocated for the hash table
                        hdestroy();
                        return -1;
                    }
                    strcpy(&list[loop_cnt * TRACE_ALD_MAXFILTERSIZE], &keybuf[loop_cnt* TRACE_ALD_MAXFILTERSIZE]);
               }
               ald_mgr->numTotalBEConfig=config_count;
            }
        }
    }
    else
    {
        TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Cannot open configuration file: %s\n", filename);
    }
    return 0;
}


int TRACE_ALD_interface_init_trace_command_structure(TRACE_ALD_Mgr *ald_mgr, int security_level_pos, int command_type,int command_pos)
{

    if(ald_mgr == NULL)
        return-1;

    switch (command_type) // Allocate memory for subcommand only as required by each TRACE command ( Number of supported subcommands)
    {
        case (TRACE_COMMAND):
        {
            ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList = (int*) malloc(sizeof(int)*TRACE_REG_CHNL_INFO);
            if((ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList) != NULL)
            {
                // Allocate memory for classConfig command only for the TRACE Command.
                ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].classCmdConfig =
                                                    (TRACE_ALD_ClassConfig*) malloc(sizeof(TRACE_ALD_ClassConfig)*TRACE_ALD_MAXFILTERS);

                if((ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].classCmdConfig) == NULL)
                {
                    free (ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList);
                    return -1;
                }
            }
            break;
        }
        case (TRACE_GET_CHNL):
        {
            ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList = (int*) malloc(sizeof(int)*TRACE_ALD_MAX_SUPPORTED_CALLBACK_CHANNELS);
            break;
        }
        case (TRACE_COMMAND_PLATFORM):
        {
            ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList = (int*) malloc(sizeof(int)*TRACE_MAX_PRXY_SZ);
            break;
        }
        default:
        {
            break;
        }
    }
    // verify if memory allocated else return an error.
    if(ald_mgr->aldConfigTable[security_level_pos].traceCmdConfig[command_pos].subCmdList == NULL)
    {
        return -1;
    }

    return 0;
}


EXPORT bool TRACE_ALD_interface_is_command_supported(void *mgr)
{
    TRACE_mgr *trace_mgr = mgr;
    bool is_command_supported = FALSE;

    if(trace_mgr == NULL)
    return is_command_supported;

    /* Process incoming request by comparing with preconfigured list of supported commands.*/
    is_command_supported = TRACE_ALD_interface_is_traceBeCmd_Supported (trace_mgr);

    return is_command_supported;
}



bool TRACE_ALD_interface_is_traceBeCmd_Supported(void *mgr)
{

    bool is_command_supported         = FALSE;
    uint8_t configured_security_level = 0;
    int traceCmdType                  = 0;
    int i                             = 0;
    uint8_t subCmd_found              = 0;
    ENTRY entrykey;
    ENTRY *entryret;
    TRACE_in_cmd_pkt* trace_cmd_pkt;
    TRACE_ALD_CmdTypeConfig *traceCfgData;
    char key[1024];

    TRACE_mgr *trace_mgr = mgr;

    if(trace_mgr == NULL)
    return FALSE;

    configured_security_level = trace_mgr->ald_mgr.system_security_level;
     // Verify security ranges for which command is supported.
    if((configured_security_level >= trace_mgr->ald_mgr.ald_customer_level) && (configured_security_level < trace_mgr->ald_mgr.ald_service_level))
    {
        is_command_supported = 0;
    }
    else if((configured_security_level >= trace_mgr->ald_mgr.ald_service_level) && (configured_security_level < trace_mgr->ald_mgr.ald_production_level))
    {
        if(trace_mgr->trc.in_msg->cmd_pkt.hdr.multi_pack == TR_MSG_NORMAL)
        {
            traceCmdType = trace_mgr->trc.in_msg->cmd_pkt.hdr.cmd_type;
            trace_cmd_pkt = &trace_mgr->trc.in_msg->cmd_pkt;

            configured_security_level = trace_mgr->ald_mgr.ald_service_level;
            // create key and search if trace command was configured for this level
            snprintf(key,TRACE_ALD_MAXFILTERSIZE,"%d:%d;", configured_security_level,traceCmdType);

            entrykey.key=key;

            entryret = hsearch(entrykey,FIND);
            if(entryret == NULL)
            {
                TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Entry not found in hash table: %s\n", key);
                return is_command_supported;
            }

            traceCfgData = (entryret->data);

            for(i=0; ((i<traceCfgData->num_sub_cmds) && (subCmd_found == FALSE)); i++)
            {
                if(trace_cmd_pkt->payld.sub_cmd == traceCfgData->subCmdList[i])
                {
                    subCmd_found = TRUE;
                    if(trace_cmd_pkt->payld.sub_cmd == TRACE_SET_CONFIG)
                    {
                        is_command_supported = TRACE_ALD_interface_is_classCfgSupported(trace_mgr,traceCfgData);
                    }
                    else if(trace_cmd_pkt->payld.sub_cmd == TRACE_CNFG_ENABLE_ALL)
                    {
                        /* special case where TR_CNFG_ENABLE_ALL has to enable only configured classes */
                        is_command_supported = TRACE_ALD_interface_cfg_ena_all(trace_mgr,traceCfgData);
                    }
                    /* Group commands are not required for service level. Since it will affect ALD requirement */
                    else if((trace_cmd_pkt->payld.sub_cmd == TRACE_SET_TRCNFG_GRP) ||
                                   (trace_cmd_pkt->payld.sub_cmd == TRACE_STOP_TRCNFG_GRP))
                    {
                        is_command_supported = FALSE;
                    }
                    else
                    {
                        is_command_supported = TRUE;
                    }
                }
             }
        }
        else if(trace_mgr->trc.in_msg->cmd_pkt.hdr.multi_pack == TR_MSG_MULTIPKT)
        {
            is_command_supported  = FALSE;
        }
        else
        {
            is_command_supported  = FALSE;
        }
    }
    else if (configured_security_level >= trace_mgr->ald_mgr.ald_production_level)
    {
       is_command_supported = TRUE;
    }
    else //Unknown Security Level.
    {
        is_command_supported = FALSE;
    }

    return is_command_supported;
}


bool TRACE_ALD_interface_is_classCfgSupported(void *mgr,TRACE_ALD_CmdTypeConfig *traceCfgData)
{
    bool is_command_supported = FALSE;
    bool class_found          = FALSE;
    TRACE_mgr* trace_mgr      = mgr;
    U8*             payld     = NULL;
    U16             val       = 0;
    U8              level     = 0;
    int i = 0;

    if((trace_mgr == NULL) ||(traceCfgData == NULL))
       return FALSE;

    payld   = trace_mgr->trc.in_msg->cmd_pkt.payld.payld;
    val     = ((U16)payld[TRACE_CLAS_OFFSET_0] << TRACE_SHIFT_8bits) | payld[TRACE_CLAS_OFFSET_1];
    level   = (payld[TRACE_CLAS_OFFSET_2]& LEVEL_MASK);

    for(i=0; ((i<traceCfgData->num_class_config) && (class_found == FALSE)); i++)
    {
        if(((traceCfgData->classCmdConfig[i].classID) == val) && ((traceCfgData->classCmdConfig[i].trace_level) >= level))
        {
            is_command_supported  = TRUE;
            class_found = TRUE;
        }
    }
    return is_command_supported;
}


bool TRACE_ALD_interface_cfg_ena_all(void *mgr,TRACE_ALD_CmdTypeConfig *traceCfgData)
{
    TRACE_mgr* trace_mgr      = mgr;
    TRACE_comp_inf* comp      = NULL;
    U8  comp_id               = 0;
    U8  clas_id               = 0;
    TRACE_level lvl           = TR_LEVEL_FATAL;
    int i                     = 0;

    if((trace_mgr == NULL) ||(traceCfgData == NULL))
       return FALSE;

    lvl = (TRACE_level)trace_mgr->trc.in_msg->cmd_pkt.payld.payld[TRACE_LVL_OFFSET];

    if(lvl <= TR_LEVEL_USER_4)
    {
        for(i=0; i<traceCfgData->num_class_config; i++)
        {
            comp_id = (((traceCfgData->classCmdConfig[i].classID) >> TRACE_SHIFT_8bits) & 0xFF);
            clas_id = ((traceCfgData->classCmdConfig[i].classID) & 0xFF);
            comp = &trace_mgr->c_cnfg->comp[comp_id];
            comp->lvl[clas_id].trc_lvl = (TRACE_level)( lvl <= (TRACE_level)traceCfgData->classCmdConfig[i].trace_level) ?
                                                           lvl : (TRACE_level)traceCfgData->classCmdConfig[i].trace_level;
        }
    }
    else
    {
        /* Invalid level */
    }

    return FALSE;
}

#endif
